---
name: project-tracker
description: Personal project management and progress tracking across mechanical, web/app, and home projects. Invoke this skill when the user wants to: (1) Start or update a project, (2) Log progress or work sessions, (3) Record decisions with rationale, (4) Manage active work vs backlog, (5) Track tasks and issues, (6) Generate status reports, (7) Sync project data to rallyai.app. Triggers include phrases like "update my project", "log progress", "what's the status of", "move to backlog", "add to active", "sync to rally".
---

# Project Tracker Skill

Track projects with clear separation between status, active work, backlog, and history. Supports dual progress tracking (planned vs overall) to accommodate discovered scope.

## Onboarding Existing Projects

Most projects won't start fresh - they'll be added mid-flight. See `references/onboarding-existing-projects.md` for full workflow.

**Quick version:**
- **Quick Capture (15 min):** Structured interview to document current state, estimate progress
- **Minimal Bootstrap (2 min):** Just start tracking from today, history builds forward
- **Deep Dive (45 min):** Full retroactive documentation

Progress for mid-project onboarding uses a **baseline model**:
- Capture estimated progress at onboarding date
- Track actual progress going forward
- Confidence improves as real data accumulates

## Core Concepts

### Document Separation

| Document | Purpose | Update Frequency |
|----------|---------|------------------|
| **STATUS.md** | High-level project health, phases, metrics | Weekly or at milestones |
| **ACTIVE.md** | Current focus - live tasks and issues | Every session |
| **BACKLOG.md** | Future work, stalled items, parking lot | As items are deferred |
| **HISTORY.md** | Completed work, sessions, decisions | Append-only log |

### Dual Progress Tracking

Projects have two progress dimensions:

```
PLANNED SCOPE (original plan)
├── Completed: 75%
└── Remaining: 25%

DISCOVERED SCOPE (emerged during work)
├── Completed: 40%
└── Remaining: 60%

OVERALL PROGRESS
├── Known work complete: 65%
└── Estimated unknowns: ~20% more scope likely
```

This acknowledges that you don't know what you don't know at project start.

## File Structure

Tracking files live in a `.tracking/` subdirectory to avoid interfering with your actual project files (CAD, code, docs, etc.).

```
~/projects/
├── .tracking/                   # Global tracking index
│   ├── index.md                 # Master project list
│   └── weekly-log.md            # Cross-project weekly summaries
│
├── cnc-machine/                 # Your actual project directory
│   ├── CAD/                     # Your existing files...
│   ├── firmware/                # Your existing files...
│   ├── docs/                    # Your existing files...
│   │
│   └── .tracking/               # Tracking subdirectory (can be gitignored)
│       ├── STATUS.md            # Project health, phases, metrics
│       ├── ACTIVE.md            # Current sprint - live work
│       ├── BACKLOG.md           # Future/stalled items
│       ├── HISTORY.md           # Completed work & session log
│       ├── decisions.md         # Decision log
│       ├── references.md        # Links, specs, docs
│       └── bom.md               # Bill of materials (mechanical)
│
├── website-project/
│   ├── src/                     # Your code...
│   └── .tracking/               # Tracking subdirectory
│       └── ...
```

**Why `.tracking/`?**
- Dot-prefix hides it in most file browsers
- Easy to add to `.gitignore` if you don't want it in version control
- Keeps tracking separate from project artifacts
- Works with any existing project structure

## Document Templates

### STATUS.md - Project Health

```markdown
# [Project Name] - Status

**Domain:** mechanical | web | home
**Health:** 🟢 On Track | 🟡 At Risk | 🔴 Blocked
**Started:** [date]
**Target:** [date or "ongoing"]
**Last Updated:** [date]

## Overview

[2-3 sentence project description]

## Progress

### Planned Scope
Work identified at project start or formally added to plan.

| Phase | Status | Progress |
|-------|--------|----------|
| Design | ✅ Complete | 100% |
| Sourcing | 🔄 Active | 80% |
| Assembly | ⏳ Pending | 0% |
| Testing | ⏳ Pending | 0% |

**Planned Progress:** 45% complete

### Discovered Scope
Work that emerged during execution (unknown unknowns).

| Item | Status | Effort |
|------|--------|--------|
| Rail alignment issues | ✅ Resolved | 4 hrs |
| Controller firmware bug | 🔄 Active | Est. 2 hrs |
| Need dust collection | ⏳ Backlog | Est. 8 hrs |

**Discovered Progress:** 30% complete
**Estimated Remaining Unknowns:** ~15% more scope likely

### Overall Progress
**Combined:** ~40% of likely total work complete

## Current Phase

**Active Phase:** Sourcing → Assembly transition
**Focus:** Complete remaining part orders, begin frame assembly
**Blockers:** Waiting on ball screws (ETA Dec 20)

## Key Metrics

| Metric | Value |
|--------|-------|
| Budget spent | $340 / $500 |
| Time invested | ~24 hrs |
| Parts received | 38 / 45 |
| Open issues | 3 |
| Decisions made | 7 |

## Recent Decisions

- **Dec 10:** Controller → MKS DLC32 (see decisions.md)
- **Dec 5:** Rail type → MGN12H (see decisions.md)

## Links

- [Repository/Docs]
- [Related Projects]
```

### ACTIVE.md - Live Working Document

```markdown
# [Project Name] - Active Work

**Current Focus:** [one-liner]
**Session:** [date]
**Sprint Goal:** [what you're trying to accomplish this cycle]

## In Progress

Active items being worked on right now.

### 🔧 Tasks

- [ ] **Wire X-axis stepper** - Started Dec 12
  - Cut wires to length ✓
  - Crimp connectors
  - Route through cable chain
  
- [ ] **Configure FluidNC** - Started Dec 12
  - Flash firmware ✓
  - Edit config for 3-axis
  - Set steps/mm

### 🐛 Issues

- [ ] **[ISS-003] Y-axis binding at 200mm travel**
  - Severity: Medium
  - Suspected: Rail alignment
  - Next step: Check with dial indicator

## Blocked

Items that cannot progress until dependency resolved.

- [ ] **Install limit switches**
  - Blocked by: Parts in transit (ETA Dec 18)
  - Ordered: Dec 8

## Today's Notes

[Running notes during work session - raw capture]

- Noticed slight wobble in Z-axis plate
- FluidNC web UI is at 192.168.1.100
- TMC2209 running cool at 1.2A

## Ready to Start

Next items to pull into active work.

- [ ] Mount controller enclosure
- [ ] Wire emergency stop
```

### BACKLOG.md - Future & Stalled Work

```markdown
# [Project Name] - Backlog

Items not currently being worked. Organized by type and priority.

## Planned Work

Original scope not yet started.

### High Priority
- [ ] **Calibration routine** - Required for accuracy
- [ ] **Spindle mount** - Needed for first cuts

### Medium Priority
- [ ] **Enclosure panels** - Dust/noise control
- [ ] **Workholding clamps** - Can use screws initially

### Low Priority
- [ ] **LED lighting** - Nice to have
- [ ] **Tool length probe** - Future upgrade

## Discovered Work

Scope that emerged during project execution.

### Should Do
- [ ] **Dust collection port** - Discovered: chips everywhere during test
- [ ] **Cable management overhaul** - Current routing is messy

### Could Do
- [ ] **Quiet cooling fans** - Stock fans are loud
- [ ] **Second E-stop** - Convenience

## Stalled / Parking Lot

Items intentionally paused or deprioritized.

- [ ] **Automatic tool changer** 
  - Reason: Scope creep, revisit after V1 working
  - Stalled: Dec 5
  
- [ ] **4th axis rotary**
  - Reason: Budget, not needed for current use case
  - Stalled: Nov 20

## Ideas / Someday

Raw ideas not yet evaluated.

- Laser attachment?
- Drag knife for vinyl?
- Probing routine for stock alignment
```

### HISTORY.md - Completed Work & Log

```markdown
# [Project Name] - History

Chronological record of completed work, sessions, and milestones.

## Milestones

| Date | Milestone | Notes |
|------|-----------|-------|
| Dec 12 | X-axis moving | First powered movement! |
| Dec 8 | Frame complete | All aluminum assembled |
| Nov 25 | Design finalized | CAD complete, BOM locked |

## Session Log

### Dec 12, 2025 - Wiring & First Movement

**Duration:** ~3 hours
**Focus:** Electrical wiring and initial motor test

#### Completed
- [x] Wired X-axis stepper motor
- [x] Flashed FluidNC firmware
- [x] First successful jog movement on X-axis

#### Discovered
- Y-axis binding issue at 200mm → Added to ACTIVE issues
- Need better cable management → Added to BACKLOG

#### Decisions
- Set motor current to 1.2A (80% of rated) for cooler operation

#### Notes
Motor runs smooth and quiet. TMC2209 StealthChop working well.
Web UI responsive. Ready to wire Y and Z tomorrow.

---

### Dec 10, 2025 - Controller Setup

**Duration:** ~2 hours
**Focus:** Controller board selection and initial setup

#### Completed
- [x] Researched controller options
- [x] Ordered MKS DLC32
- [x] Downloaded FluidNC

#### Decisions
- Selected MKS DLC32 over Arduino CNC Shield (see decisions.md)

---

## Completed Items Archive

Items moved from ACTIVE when finished.

### Tasks Completed
- [x] **Order linear rails** - Completed Dec 1
- [x] **Assemble frame** - Completed Dec 8
- [x] **Mount X-axis rails** - Completed Dec 10

### Issues Resolved
- [x] **[ISS-001] Rail alignment** - Resolved Dec 10
  - Solution: Added 0.1mm shims under rail
- [x] **[ISS-002] Wrong screws delivered** - Resolved Dec 5
  - Solution: Vendor shipped replacement
```

## Workflows

### Moving Items Between Documents

**ACTIVE → HISTORY** (completed)
```
When task/issue is done:
1. Mark complete in ACTIVE.md
2. Move to HISTORY.md "Completed Items Archive"
3. Update STATUS.md progress metrics
4. Sync to Rally: PATCH /api/tasks/{id} status=done
```

**ACTIVE → BACKLOG** (stalled/deferred)
```
When item is blocked long-term or deprioritized:
1. Remove from ACTIVE.md
2. Add to BACKLOG.md appropriate section
3. Note reason and date stalled
4. Sync to Rally: PATCH /api/tasks/{id} status=backlog
```

**BACKLOG → ACTIVE** (starting work)
```
When pulling item into active work:
1. Remove from BACKLOG.md
2. Add to ACTIVE.md "In Progress" or "Ready to Start"
3. Sync to Rally: PATCH /api/tasks/{id} status=active
```

**New discovered work**
```
When unknown work emerges:
1. Add to ACTIVE.md if immediate, else BACKLOG.md
2. Add to STATUS.md "Discovered Scope" table
3. Adjust "Estimated Remaining Unknowns" if needed
4. Sync to Rally: POST /api/tasks with scope_type=discovered
```

### Session Workflow

**Starting a session:**
1. Read STATUS.md for project health
2. Read ACTIVE.md for current focus
3. Pull context from Rally: GET /api/sync/context/{project}
4. Ask: "Continuing [focus]. Any blockers resolved?"

**During session:**
1. Update ACTIVE.md "Today's Notes" with observations
2. Check off completed subtasks
3. Note any discovered issues

**Ending session:**
1. Move completed items ACTIVE → HISTORY
2. Move stalled items ACTIVE → BACKLOG
3. Update STATUS.md metrics if significant progress
4. Write session summary in HISTORY.md
5. Sync to Rally: POST /api/sync with full session data

### Progress Calculation

```python
# Planned progress
planned_total = sum(planned_item.effort for all planned items)
planned_done = sum(item.effort for completed planned items)
planned_pct = planned_done / planned_total

# Discovered progress  
discovered_total = sum(discovered_item.effort for all discovered items)
discovered_done = sum(item.effort for completed discovered items)
discovered_pct = discovered_done / discovered_total if discovered_total > 0 else N/A

# Overall (weighted by effort)
total_known = planned_total + discovered_total
done_known = planned_done + discovered_done
overall_known_pct = done_known / total_known

# Unknown estimate (heuristic based on project phase and history)
# Early project: estimate 30-50% more scope will emerge
# Mid project: estimate 15-25% more
# Late project: estimate 5-15% more
estimated_unknowns_pct = estimate_based_on_phase()
```

## Quick Reference

| Action | Document | Rally Endpoint |
|--------|----------|----------------|
| Start task | ACTIVE.md | POST /api/rallyai/projects/{id}/tasks |
| Complete task | ACTIVE→HISTORY | POST /api/rallyai/projects/{id}/tasks/{taskId}/complete |
| Update task | ACTIVE.md | PATCH /api/rallyai/projects/{id}/tasks/{taskId} |
| Delete task | - | DELETE /api/rallyai/projects/{id}/tasks/{taskId} |
| Log issue | ACTIVE.md | POST /api/rallyai/projects/{id}/issues |
| Resolve issue | ACTIVE→HISTORY | PATCH /api/rallyai/projects/{id}/issues/{issueId} |
| Update progress | STATUS.md | PATCH /api/rallyai/projects/{id} |
| Log session | HISTORY.md | POST /api/rallyai/projects/{id}/sessions |
| Log decision | decisions.md | POST /api/rallyai/projects/{id}/decisions |

## CLI Commands (rally_sync.py)

```bash
# List tasks for a project
rally_sync.py task-list --project <PROJECT_ID> [--status todo|in-progress|completed]

# Create a new task
rally_sync.py task-create --project <PROJECT_ID> --title "Task name" \
  [--description "Details"] [--priority high] [--status todo] [--estimate 60]

# Update a task
rally_sync.py task-update --project <PROJECT_ID> --task <TASK_ID> \
  [--status in-progress] [--priority urgent] [--title "New title"] [--estimate 90]

# Complete a task
rally_sync.py task-complete --project <PROJECT_ID> --task <TASK_ID> [--time 30]

# Delete a task
rally_sync.py task-delete --project <PROJECT_ID> --task <TASK_ID>

# Log a work session
rally_sync.py session --project <PROJECT_ID> --summary "What was done" \
  --accomplishments "Item 1" "Item 2" --duration 60

# Log a decision
rally_sync.py decision --project <PROJECT_ID> --title "Decision title" \
  --decision "What was decided" --rationale "Why"

# Update project progress
rally_sync.py progress --project <PROJECT_ID> --planned 50 --discovered 20
```

## Behavior Guidelines

1. **Keep ACTIVE lean:** Only items being worked NOW or this week
2. **BACKLOG is not a graveyard:** Review monthly, archive or delete stale items
3. **STATUS reflects reality:** Update weekly minimum, be honest about health
4. **HISTORY is append-only:** Never edit past entries, only add
5. **Track discovered scope:** This is valuable project intelligence
6. **Estimate unknowns:** Get better at this over time by comparing actuals
