#!/usr/bin/env python3
"""
Rally Sync Script v3
Single project sync to rallyai.app
"""

import argparse
import json
import os
import sys
from datetime import datetime
from pathlib import Path
import urllib.request
import urllib.error

CONFIG_PATH = Path.home() / ".config" / "rallyai" / "credentials.json"


def load_config():
    """Load API credentials."""
    if not CONFIG_PATH.exists():
        raise FileNotFoundError(
            f"Credentials not found at {CONFIG_PATH}\n"
            "Create with: {\"api_key\": \"rally_...\", \"base_url\": \"https://...\"}"
        )
    with open(CONFIG_PATH) as f:
        return json.load(f)


def api_request(method, endpoint, config, data=None):
    """Make authenticated API request."""
    url = f"{config['base_url']}{endpoint}"
    headers = {
        "Authorization": f"Bearer {config['api_key']}",
        "Content-Type": "application/json",
    }
    
    body = json.dumps(data).encode() if data else None
    req = urllib.request.Request(url, data=body, headers=headers, method=method)
    
    try:
        with urllib.request.urlopen(req, timeout=30) as response:
            return json.loads(response.read().decode())
    except urllib.error.HTTPError as e:
        error_body = e.read().decode() if e.fp else ""
        raise RuntimeError(f"API Error {e.code}: {error_body}")
    except urllib.error.URLError as e:
        raise RuntimeError(f"Connection Error: {e.reason}")


def get_context(project_id, config):
    """Get project context for session start."""
    return api_request("GET", f"/rallyai/sync/context/{project_id}", config)


def sync_session(project_id, session_file, config):
    """Sync session data to Rally."""
    with open(session_file) as f:
        data = json.load(f)
    
    payload = {
        "project_id": project_id,
        "sync_type": "session_end",
        "timestamp": datetime.utcnow().isoformat() + "Z",
        "data": data
    }
    
    return api_request("POST", "/rallyai/sync", config, payload)


def update_progress(project_id, planned_pct, discovered_pct, config):
    """Update project progress."""
    return api_request("PATCH", f"/rallyai/projects/{project_id}", config, {
        "progress_percentage": planned_pct
    })


def log_decision(project_id, title, decision, rationale, config):
    """Log a decision."""
    return api_request("POST", f"/rallyai/projects/{project_id}/decisions", config, {
        "title": title,
        "decision": decision,
        "rationale": rationale,
        "date": datetime.utcnow().strftime("%Y-%m-%d")
    })


def log_session(project_id, summary, accomplishments, next_steps, duration_min, config):
    """Log a work session."""
    return api_request("POST", f"/rallyai/projects/{project_id}/sessions", config, {
        "date": datetime.utcnow().strftime("%Y-%m-%d"),
        "duration_minutes": duration_min,
        "summary": summary,
        "accomplishments": accomplishments,
        "next_steps": next_steps
    })


# Task management functions
def list_tasks(project_id, status, config):
    """List tasks for a project."""
    endpoint = f"/rallyai/projects/{project_id}/tasks"
    if status:
        endpoint += f"?status={status}"
    return api_request("GET", endpoint, config)


def create_task(project_id, title, description, priority, status, estimate, config):
    """Create a new task."""
    data = {"title": title}
    if description:
        data["description"] = description
    if priority:
        data["priority"] = priority
    if status:
        data["status"] = status
    if estimate:
        data["estimated_time"] = {"amount": estimate, "unit": "minutes"}
    return api_request("POST", f"/rallyai/projects/{project_id}/tasks", config, data)


def update_task(project_id, task_id, updates, config):
    """Update an existing task."""
    return api_request("PATCH", f"/rallyai/projects/{project_id}/tasks/{task_id}", config, updates)


def complete_task(project_id, task_id, actual_time, config):
    """Mark a task as complete."""
    data = {}
    if actual_time:
        data["actual_time"] = {"amount": actual_time, "unit": "minutes"}
    return api_request("POST", f"/rallyai/projects/{project_id}/tasks/{task_id}/complete", config, data)


def delete_task(project_id, task_id, config):
    """Delete a task."""
    return api_request("DELETE", f"/rallyai/projects/{project_id}/tasks/{task_id}", config)


def main():
    parser = argparse.ArgumentParser(description="Rally Sync v3 - Single Project")
    subparsers = parser.add_subparsers(dest="command", required=True)
    
    # context
    ctx = subparsers.add_parser("context", help="Get project context")
    ctx.add_argument("--project", required=True, help="Project ID")
    
    # sync
    sync = subparsers.add_parser("sync", help="Sync session from JSON file")
    sync.add_argument("--project", required=True, help="Project ID")
    sync.add_argument("--file", type=Path, required=True, help="Session JSON")
    
    # progress
    prog = subparsers.add_parser("progress", help="Update progress")
    prog.add_argument("--project", required=True)
    prog.add_argument("--planned", type=int, required=True, help="Planned %")
    prog.add_argument("--discovered", type=int, required=True, help="Discovered %")
    
    # decision
    dec = subparsers.add_parser("decision", help="Log a decision")
    dec.add_argument("--project", required=True)
    dec.add_argument("--title", required=True)
    dec.add_argument("--decision", required=True)
    dec.add_argument("--rationale", required=True)
    
    # session
    sess = subparsers.add_parser("session", help="Log a session")
    sess.add_argument("--project", required=True)
    sess.add_argument("--summary", required=True)
    sess.add_argument("--accomplishments", nargs="+", default=[])
    sess.add_argument("--next-steps", nargs="+", default=[])
    sess.add_argument("--duration", type=int, default=60, help="Minutes")

    # task-list
    tl = subparsers.add_parser("task-list", help="List tasks for a project")
    tl.add_argument("--project", required=True, help="Project ID")
    tl.add_argument("--status", help="Filter by status (collect/todo/in-progress/completed/blocked)")

    # task-create
    tc = subparsers.add_parser("task-create", help="Create a new task")
    tc.add_argument("--project", required=True, help="Project ID")
    tc.add_argument("--title", required=True, help="Task title")
    tc.add_argument("--description", help="Task description")
    tc.add_argument("--priority", choices=["low", "medium", "high", "urgent"], default="medium")
    tc.add_argument("--status", choices=["collect", "todo", "in-progress", "blocked"], default="todo")
    tc.add_argument("--estimate", type=int, help="Estimated time in minutes")

    # task-update
    tu = subparsers.add_parser("task-update", help="Update a task")
    tu.add_argument("--project", required=True, help="Project ID")
    tu.add_argument("--task", required=True, help="Task ID")
    tu.add_argument("--title", help="New title")
    tu.add_argument("--description", help="New description")
    tu.add_argument("--status", choices=["collect", "todo", "in-progress", "completed", "blocked"])
    tu.add_argument("--priority", choices=["low", "medium", "high", "urgent"])
    tu.add_argument("--estimate", type=int, help="Estimated time in minutes")

    # task-complete
    tcomp = subparsers.add_parser("task-complete", help="Mark a task as complete")
    tcomp.add_argument("--project", required=True, help="Project ID")
    tcomp.add_argument("--task", required=True, help="Task ID")
    tcomp.add_argument("--time", type=int, help="Actual time spent in minutes")

    # task-delete
    td = subparsers.add_parser("task-delete", help="Delete a task")
    td.add_argument("--project", required=True, help="Project ID")
    td.add_argument("--task", required=True, help="Task ID")

    args = parser.parse_args()
    
    try:
        config = load_config()
    except FileNotFoundError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
    
    try:
        if args.command == "context":
            result = get_context(args.project, config)
            print(json.dumps(result, indent=2))
            
        elif args.command == "sync":
            result = sync_session(args.project, args.file, config)
            print(f"Synced: {result.get('session', {}).get('id', 'OK')}")
            
        elif args.command == "progress":
            result = update_progress(args.project, args.planned, args.discovered, config)
            print(f"Progress updated")
            
        elif args.command == "decision":
            result = log_decision(args.project, args.title, args.decision, args.rationale, config)
            print(f"Decision logged: {result.get('id', 'OK')}")
            
        elif args.command == "session":
            result = log_session(
                args.project, args.summary, args.accomplishments,
                args.next_steps, args.duration, config
            )
            print(f"Session logged: {result.get('id', 'OK')}")

        elif args.command == "task-list":
            result = list_tasks(args.project, args.status, config)
            tasks = result.get("tasks", [])
            if not tasks:
                print("No tasks found")
            else:
                for t in tasks:
                    status_icon = {"collect": "📥", "todo": "📋", "in-progress": "🔄", "completed": "✅", "blocked": "🚫"}.get(t["status"], "•")
                    est = t.get("estimated_time")
                    est_str = f" ~{est['amount']}min" if est and est.get("amount") else ""
                    print(f"{status_icon} [{t['id'][:8]}] {t['title']} ({t['status']}){est_str}")

        elif args.command == "task-create":
            result = create_task(
                args.project, args.title, args.description,
                args.priority, args.status, args.estimate, config
            )
            est = result.get('estimated_time')
            est_str = f" ({est['amount']} min)" if est and est.get('amount') else ""
            print(f"Task created: {result.get('id', 'OK')} - {result.get('title', '')}{est_str}")

        elif args.command == "task-update":
            updates = {}
            if args.title:
                updates["title"] = args.title
            if args.description:
                updates["description"] = args.description
            if args.status:
                updates["status"] = args.status
            if args.priority:
                updates["priority"] = args.priority
            if args.estimate:
                updates["estimated_time"] = {"amount": args.estimate, "unit": "minutes"}
            if not updates:
                print("No updates specified", file=sys.stderr)
                sys.exit(1)
            result = update_task(args.project, args.task, updates, config)
            print(f"Task updated: {result.get('id', 'OK')} - {result.get('status', '')}")

        elif args.command == "task-complete":
            result = complete_task(args.project, args.task, args.time, config)
            print(f"Task completed: {result.get('id', 'OK')} - {result.get('title', '')}")

        elif args.command == "task-delete":
            result = delete_task(args.project, args.task, config)
            print(f"Task deleted")

    except RuntimeError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
